/*==============================================================================
PART 2E: Bank Value Calculation
==============================================================================
Purpose:
This script calculates different measures of bank value relative to assets,
incorporating the estimated mark-to-market asset losses and deposit franchise
values calculated in previous steps. The values are calculated for specific
quarters relative to a base period equity-to-assets ratio.

Input:
- $path_clean/call_reports_prepared.dta (Prepared bank-quarter panel from Part 1)
- $path_temp/deposit_franchises${ext_suffix}.dta (Deposit franchise values from Part 2C)
- $path_temp/mtm_assets${ext_suffix}.dta (Mark-to-market asset losses from Part 2D)

Output:
- A Stata dataset in long panel form containing bank-period observations.
  Each observation identifies `rssdid`, `yq` (the end quarter of the period),
  and `period` (the name of the estimation period). It includes different
  measures of bank value (`v_0`, `v_1`, `v_noDF`) for that specific period,
  expressed as a percentage of base quarter assets.
  Example structure (simplified):
     +-----------------------------------------------------------------+
     | rssdid       yq      period      v_0      v_1    v_noDF |
     |-----------------------------------------------------------------|
  1. |   6329   2021q4   dec2021     15.50    18.30     12.00 |
  2. |   6329   2022q4   feb2023     10.20    12.80      8.50 |
  3. |   6329   2023q4   feb2024     11.40    14.10      9.20 |
  4. |   6330   2021q4   dec2021     12.00    15.00     10.00 |
  5. |   6330   2022q4   feb2023      9.00    11.50      7.50 |
     |-----------------------------------------------------------------|
  6. |   6330   2023q4   feb2024     10.10    13.00      8.80 |
     ......continued .....

Methodology:
1. Load the prepared bank panel and apply sample restrictions.
2. Merge in estimated deposit franchise values and MTM losses.
3. Determine a base equity-to-assets ratio, prioritizing 2021q4, then 2022q1,
   then 2021q1 for banks missing in the primary base quarter.
4. Calculate different bank value measures using vectorized formulas, relative
   to the base equity-to-assets ratio and incorporating asset losses and
   deposit franchise values.
5. Handle missing franchise values by implicitly treating them as zero in calculations.
6. Save the resulting long format dataset.

Last updated: July 25, 2025
==============================================================================*/

display "--- Starting Part 2E: Bank Value Calculation ---" // Indicate the start of the script

clear all

use "$path_clean/call_reports_prepared.dta", clear 

// Define the end quarters for the estimation periods
local dt_2021q4 = tq(2021q4) // dec2021 period end
local dt_2022q4 = tq(2022q4) // feb2023 period end
local dt_2023q4 = tq(2023q4) // feb2024 period end

// Create a string variable to identify the period for each observation corresponding to an end quarter
gen period = ""
replace period = "dec2021" if yq == `dt_2021q4'
replace period = "feb2023" if yq == `dt_2022q4'
replace period = "feb2024" if yq == `dt_2023q4'

sort rssdid yq // Sort by bank and quarter

//===============================================================================
// Step 1: Merge Estimated Values
//===============================================================================
// Purpose: Merge in estimated deposit franchise values and MTM losses, which are
// in long format (by rssdid and period). This requires a temporary reshape/merge
// approach to align the long format data with the bank-quarter panel.
// Note: The ${ext_suffix} global is used to differentiate between different
// extensions in section 04, which reuse this code.

// Preserve the current bank-quarter dataset
preserve
// Keep only observations with a defined period (the end quarters) before merging long format data
keep if period != ""

// Merge in deposit franchise values from Part 2C (by rssdid and period)
merge 1:1 rssdid period using "$path_temp/deposit_franchises${ext_suffix}.dta", keep(master match) nogen

// Merge in mark-to-market asset losses from Part 2D (by rssdid and period)
merge 1:1 rssdid period using "$path_temp/mtm_assets${ext_suffix}.dta", keep(master match) nogen

sort rssdid period // Sort the merged long-format data
tempfile temp_merge 
save `temp_merge', replace // Save the merged long-format data to the tempfile
restore // Restore the original bank-quarter dataset

// Merge the long-format data from the tempfile back into the main bank-quarter dataset using rssdid and yq
// This brings the period-specific estimates (franchise values, losses) into the correct quarters.
merge 1:1 rssdid yq using `temp_merge', keep(master match) nogen

//===============================================================================
// Step 2: Determine Base Equity-to-Assets Ratio
//===============================================================================
// Purpose: Establish a base equity-to-assets ratio for each bank, primarily
// using the value from 2021q4. If 2021q4 is missing, use 2022q1, and if that
// is also missing, use 2021q1. This base value is then propagated across all
// periods for the bank.

// Conditionally adjust assets and equity values for capital requirements extensions
// if the global ${cap_rule} is set. This modifies the balance sheet items
// before calculating the base equity-to-assets ratio.
if "${cap_rule}" != "" {
  display "Adjusting assets and equity for capital requirements based on cap_rule: ${cap_rule}"
  replace assets = assets + ${cap_rule}
  replace equity = equity + ${cap_rule}
  replace equity_assets = equity / assets // Recalculate equity-to-assets ratio after adjustment
}

// Calculate the equity-to-assets ratio in the primary base quarter (2021q4)
gen equity_assets_fv = equity_assets if yq == tq(2021q4)
// Propagate the 2021q4 value across all quarters for each bank
bys rssdid: egen temp_equity = max(equity_assets_fv)
replace equity_assets_fv = temp_equity
drop temp_equity

// If 2021q4 value is missing, fall back to the value in the alternative base quarter (2022q1)
replace equity_assets_fv = equity_assets if yq == tq(2022q1) & missing(equity_assets_fv)
// Propagate the 2022q1 value for banks where 2021q4 was missing
bys rssdid: egen temp_equity = max(equity_assets_fv)
replace equity_assets_fv = temp_equity
drop temp_equity

// If both 2021q4 and 2022q1 values are missing, fall back to the value in 2021q1
replace equity_assets_fv = equity_assets if yq == tq(2021q1) & missing(equity_assets_fv)
// Propagate the 2021q1 value for banks where 2021q4 and 2022q1 were missing
bys rssdid: egen temp_equity = max(equity_assets_fv)
replace equity_assets_fv = temp_equity
drop temp_equity

// Restrict the dataset to keep only the relevant end quarters for the final output
keep if yq == `dt_2021q4' | yq == `dt_2022q4' | yq == `dt_2023q4'

//===============================================================================
// Step 3: Calculate Bank Value Measures
//===============================================================================
// Purpose: Calculate different measures of bank value relative to assets,
// expressed as a percentage of base quarter assets.

// Handle missing losses: set to 0 for the dec2021 period (base period),
// use the calculated actual loss for other periods.
replace loss = 0 if period == "dec2021" & missing(loss)

// v_noDF: Bank value excluding deposit franchise.
// Formula: Base Equity-to-Assets Ratio (%) - Asset Loss (%)
gen v_noDF = equity_assets_fv * 100 - loss

// v_0: Bank value including deposit franchise, adjusted for assumed deposit run-off.
// Formula: v_noDF + (1 - run_i) * Insured Deposit Franchise Value + (1 - run_u) * Uninsured Deposit Franchise Value
// run_i and run_u are globals representing the portion of insured and uninsured deposits assumed to run off.
gen v_0 = v_noDF + (1-${run_i})*DI_value + (1-${run_u})*DU_value

// v_1: Bank value including the full estimated deposit franchise value.
// Formula: v_noDF + Total Deposit Franchise Value (Insured + Uninsured)
gen v_1 = v_noDF + DI_value + DU_value

//===============================================================================
// Step 4: Handle Missing Franchise Values
//===============================================================================
// Purpose: For banks where DI_value or DU_value could not be calculated (e.g.,
// due to missing betas or costs), the formulas in Step 3 would result in missing
// v_0 or v_1. This step implicitly handles this by setting v_0 or v_1 based on
// the available components, effectively treating missing franchise values as 0.

// If DI_value is missing, recalculate v_0 using only the DU_value component.
// This treats the missing DI_value as zero.
replace v_0 = v_noDF + (1-${run_u})*DU_value if missing(DI_value)

// If DU_value is missing, recalculate v_1 using only the DI_value component (and v_noDF).
// This treats the missing DU_value as zero.
replace v_1 = v_noDF + DI_value if missing(DU_value) // Note: This is equivalent to v_0 + DI_value if DU_value is missing

//===============================================================================
// Step 5: Save Final Dataset
//===============================================================================
// Purpose: Keep only the identifier variables and the newly created bank
// value variables and save the dataset to the temporary directory.

// Keep only relevant variables for the final output dataset
keep rssdid period v_0 v_1 v_noDF

// Sort the final dataset
sort rssdid period

// Save the dataset
save "$path_temp/bank_values${ext_suffix}.dta", replace

display "--- Bank Value Calculation completed ---" // Indicate the completion of the script

